//
//  TodoItemDetail.swift
//  Do It
//
//  Created by Jim Dovey on 9/9/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoItemDetail: View {
    let item: TodoItem
    
    var gradient: LinearGradient {
        LinearGradient(
            gradient: Gradient(colors: [
                Color.black.opacity(0.6),
                Color.black.opacity(0),
            ]),
            startPoint: .bottom,
            endPoint: .init(x: 0.5, y: 0.1)
        )
    }
    
    static var formatter: DateFormatter = {
        let f = DateFormatter()
        f.dateStyle = .medium
        f.timeStyle = .none
        return f
    }()

    var body: some View {
        ScrollView(.vertical) {
            VStack(alignment: .leading) {
                // START:ResizableHeader
                VStack(alignment: .leading, spacing: 8) {
                    Text(verbatim: item.title)
                        .font(.title)
                        .bold()
                        .layoutPriority(1)
                    
                    // <literal:elide> … more … </literal:elide>
                    // END:ResizableHeader
                    HStack(alignment: .firstTextBaseline) {
                        // START:LocalizedStringKey
                        Text("Priority: ") +
                            Text(LocalizedStringKey(item.priority.rawValue.capitalized)).bold()
                        // END:LocalizedStringKey
                        Spacer()
                            .layoutPriority(-1)
                        if item.date != nil {
                            Text("\(item.date!, formatter: Self.formatter)")
                        } else {
                            Text("No Due Date")
                        }
                    }
                    .font(.subheadline)
                    // START:ResizableHeader
                }
                .foregroundColor(.white)
                .padding()
                .padding(.top)
                    // START_HIGHLIGHT
                    .background(ZStack {
                        Rectangle()
                            .fill(item.list.color.uiColor)
                            .overlay(gradient)
                            .edgesIgnoringSafeArea(.top)
                    })
                    // END_HIGHLIGHT
                    // END:ResizableHeader
                    .layoutPriority(1)
                // START:ResizableHeader
                
                if item.notes != nil {
                    Text(verbatim: self.item.notes!)
                        .padding()
                }
                Spacer()
                    // END:ResizableHeader
                    .layoutPriority(-1)
            }
        }
        .navigationBarTitle("", displayMode: .inline)
    }
}

struct TodoItemDetail_Previews: PreviewProvider {
    static var previews: some View {
        single
    }
    
    static var single: some View {
        NavigationView {
            TodoItemDetail(item: todoItems[0])
                .navigationBarItems(leading: HStack(spacing: 4) {
                    Image(systemName: "chevron.left").imageScale(.large).font(.headline)
                    Text("To-Do Items")
                }.foregroundColor(.accentColor))
        }
    }
    
    static var allColorSchemePreviews: some View {
        // START:ColorSchemes
        ForEach(ColorScheme.allCases, id: \.self) { scheme in
            NavigationView {
                TodoItemDetail(item: todoItems[0])
            }
            .colorScheme(scheme)
            .previewDisplayName(String(describing: scheme))
        }
        // END:ColorSchemes
    }
    
    static var multiDevicePreviews: some View {
        // START:MultipleDevices
        ForEach(["iPhone 11", "iPhone SE"], id: \.self) { name in
            TodoItemDetail(item: todoItems[0])
                .previewDevice(PreviewDevice(rawValue: name))
                .previewDisplayName(name)
        }
        // END:MultipleDevices
    }
    
    static var allItemPreviews: some View {
        // START:AllPreviewItems
        ForEach(todoItems) { item in
            TodoItemDetail(item: item)
        }
        // END:AllPreviewItems
    }
    
    static var allLocalizations: some View {
        // START:Localizations
        let localePairs: [(Locale, LayoutDirection)] = [
            (Locale(identifier: "en-US"), .leftToRight),
            (Locale(identifier: "pl"), .leftToRight),
            (Locale(identifier: "ar"), .rightToLeft),
        ]
        
        return ForEach(localePairs, id: \.self.0) { value in
            TodoItemDetail(item: todoItems[0])
                .environment(\.locale, value.0)
                .environment(\.layoutDirection, value.1)
                .previewDisplayName(value.0.languageCode ?? value.0.identifier)
        }
        // END:Localizations
    }
    
    static var allSizeCategories: some View {
        // START:SizeCategories
        ForEach(ContentSizeCategory.allCases, id: \.self) { category in
            TodoItemDetail(item: todoItems[0])
                .environment(\.sizeCategory, category)
                .previewDisplayName(String(describing: category))
        }
        // END:SizeCategories
    }
}
